#include <SPI.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define SCREEN_ADDRESS 0x3C
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, -1);

#define BTN_LEFT     9
#define BTN_RIGHT    7
#define BTN_UP       10
#define BTN_DOWN     8
#define BTN_FIRE     5

const uint8_t PROGMEM spaceshipBitmap[] = {
  0b00110000,
  0b00110000,
  0b01111000,
  0b00110000,
  0b01001000,
  0b10000100
};

const uint8_t PROGMEM enemyBitmap[] = {
  0b01111000,
  0b10010100,
  0b11111100,
  0b10010100,
  0b10000100,
  0b01001000
};

const uint8_t PROGMEM alienSkull36x28[] = {
  0x00, 0xFF, 0xFF, 0xF0, 0x00, 0x00, 0xFF, 0xFF, 0xF0, 0x00, 0x00, 0xFF,
  0xFF, 0xF0, 0x00, 0x00, 0xFF, 0xFF, 0xF0, 0x00, 0x0F, 0x00, 0x00, 0x0F,
  0x00, 0x0F, 0x00, 0x00, 0x0F, 0x00, 0x0F, 0x00, 0x00, 0x0F, 0x00, 0x0F,
  0x00, 0x00, 0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x0F, 0x0F, 0x0F,
  0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00,
  0x0F, 0x00, 0x00, 0x0F, 0x00, 0x0F, 0x00, 0x00, 0x0F, 0x00, 0x0F, 0x00,
  0x00, 0x0F, 0x00, 0x0F, 0x00, 0x00, 0x0F, 0x00, 0x0F, 0xFF, 0xFF, 0xFF,
  0x00, 0x0F, 0xFF, 0xFF, 0xFF, 0x00, 0x0F, 0xFF, 0xFF, 0xFF, 0x00, 0x0F,
  0xFF, 0xFF, 0xFF, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x0F, 0x0F, 0x0F,
  0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00,
  0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x0F, 0x0F,
  0x0F, 0x0F, 0x00, 0x0F, 0x0F, 0x0F, 0x0F, 0x00,
};

int playerX = 56, playerY = 30;
int movementSpeed = 2;
int score = 0;
int lives = 3;
bool showDamage = false;
bool gameOver = false;
bool inGame = false;

unsigned long damageTime = 0;
unsigned long lastFireTime = 0;
unsigned long lastSpawnTime = 0;
unsigned long lastSpeedUpTime = 0;
unsigned long spawnInterval = 2000;
int difficulty = 1;

const int maxBullets = 3;
const int maxEnemies = 7;
struct Bullet { int x, y; bool active; };
struct Enemy { int x, y; bool active; };
Bullet bullets[maxBullets];
Enemy enemies[maxEnemies];

struct Controls {
  bool up;
  bool down;
  bool left;
  bool right;
  bool fire;
};

Controls readControls() {
  Controls c;
  c.left  = !digitalRead(BTN_LEFT);
  c.right = !digitalRead(BTN_RIGHT);
  c.up    = !digitalRead(BTN_UP);
  c.down  = !digitalRead(BTN_DOWN);
  c.fire  = !digitalRead(BTN_FIRE);
  return c;
}

void setup() {
  Serial.begin(9600);
  pinMode(BTN_LEFT, INPUT_PULLUP);
  pinMode(BTN_RIGHT, INPUT_PULLUP);
  pinMode(BTN_UP, INPUT_PULLUP);
  pinMode(BTN_DOWN, INPUT_PULLUP);
  pinMode(BTN_FIRE, INPUT_PULLUP);
  lastSpeedUpTime = millis();
  if (!display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) while (true);
  display.setRotation(1);
  display.clearDisplay();
  display.display();
}

void applyDifficulty(int d) {
  if (d == 0) { movementSpeed = 3; spawnInterval = 3000; }
  if (d == 1) { movementSpeed = 2; spawnInterval = 2000; }
  if (d == 2) { movementSpeed = 1; spawnInterval = 1200; }
}

void showMenu() {
  const char* options[] = { "Easy", "Normal", "Hard" };
  while (!inGame) {
    display.clearDisplay();
    display.setTextSize(1);
    display.setTextColor(SSD1306_WHITE);
    display.setCursor(0, 0);
    display.println("Select");
    display.setCursor(0, 10);
    display.println("Difficulty");

    for (int i = 0; i < 3; i++) {
      display.setCursor(0, 20 + i * 10);
      display.print(i == difficulty ? "> " : "  ");
      display.println(options[i]);
    }

    display.setCursor(0, 55);
    display.println("Press FIRE to Start");
    display.display();

    Controls c = readControls();
    if (c.up) { difficulty = (difficulty + 2) % 3; delay(200); }
    if (c.down) { difficulty = (difficulty + 1) % 3; delay(200); }
    if (c.fire) {
      applyDifficulty(difficulty);
      inGame = true;
      gameOver = false;
      playerX = 56;
      playerY = 30;
      score = 0;
      lives = 3;
      resetGameObjects();
      delay(300);
    }
  }
}

void handleInput() {
  Controls c = readControls();
  if (c.up && playerX > 0) playerX -= movementSpeed;
  if (c.down && playerX < (SCREEN_WIDTH - 6)) playerX += movementSpeed;
  if (c.left && playerY > 0) playerY -= movementSpeed;
  if (c.right && playerY < (SCREEN_HEIGHT - 6)) playerY += movementSpeed;
  if (c.fire) fireBullet();
}

void fireBullet() {
  if (millis() - lastFireTime < 400) return;
  for (int i = 0; i < maxBullets; i++) {
    if (!bullets[i].active) {
      bullets[i] = { playerX - 2, playerY + 3, true };
      lastFireTime = millis();
      break;
    }
  }
}

void updateBullets() {
  for (int i = 0; i < maxBullets; i++) {
    if (bullets[i].active) {
      bullets[i].x -= 4;
      if (bullets[i].x < 0) bullets[i].active = false;
    }
  }
}

void spawnEnemy() {
  if (millis() - lastSpawnTime < spawnInterval) return;
  for (int i = 0; i < maxEnemies; i++) {
    if (!enemies[i].active) {
      enemies[i] = { 0, random(0, SCREEN_HEIGHT - 6), true };
      break;
    }
  }
  if (millis() - lastSpeedUpTime > 3000 && spawnInterval > 600) {
    spawnInterval -= 100;
    lastSpeedUpTime = millis();
  }
  lastSpawnTime = millis();
}

void updateEnemies() {
  for (int i = 0; i < maxEnemies; i++) {
    if (enemies[i].active) {
      enemies[i].x += 1;
      if (enemies[i].x > SCREEN_WIDTH) {
        enemies[i].active = false;
        lives--;
        showDamage = true;
        damageTime = millis();
        if (lives <= 0) gameOver = true;
      }
    }
  }
}

void checkCollisions() {
  for (int b = 0; b < maxBullets; b++) {
    if (!bullets[b].active) continue;
    for (int e = 0; e < maxEnemies; e++) {
      if (!enemies[e].active) continue;
      if (bullets[b].x < enemies[e].x + 6 &&
          bullets[b].x + 2 > enemies[e].x &&
          bullets[b].y < enemies[e].y + 6 &&
          bullets[b].y + 2 > enemies[e].y) {
        bullets[b].active = false;
        enemies[e].active = false;
        score += 10;
        break;
      }
    }
  }

  for (int e = 0; e < maxEnemies; e++) {
    if (!enemies[e].active) continue;
    if (playerX < enemies[e].x + 6 &&
        playerX + 6 > enemies[e].x &&
        playerY < enemies[e].y + 6 &&
        playerY + 6 > enemies[e].y) {
      enemies[e].active = false;
      showDamage = true;
      damageTime = millis();
      lives--;
      if (lives <= 0) gameOver = true;
    }
  }
}

void drawPlayer() {
  display.drawBitmap(playerY, playerX, spaceshipBitmap, 6, 6, SSD1306_WHITE);
}

void drawBullets() {
  for (int i = 0; i < maxBullets; i++)
    if (bullets[i].active)
      display.drawRect(bullets[i].y, bullets[i].x, 2, 2, SSD1306_WHITE);
}

void drawEnemies() {
  for (int i = 0; i < maxEnemies; i++)
    if (enemies[i].active)
      display.drawBitmap(enemies[i].y, enemies[i].x, enemyBitmap, 6, 6, SSD1306_WHITE);
}

void drawHUD() {
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  display.println(score);
  for (int i = 0; i < lives; i++) {
    display.setCursor(SCREEN_HEIGHT - 6, SCREEN_WIDTH - 8 - (i * 6));
    display.write('+');
  }
}

void drawDamageEffect() {
  if (showDamage && millis() - damageTime < 150) {
    for (int y = 0; y < SCREEN_HEIGHT; y++) {
      display.drawPixel(126, y, SSD1306_WHITE);
    }
  } else {
    showDamage = false;
  }
}

void showGameOverScreen() {
  display.clearDisplay();
  int skullX = (SCREEN_HEIGHT - 36) / 2;
  int skullY = 6;
  display.drawBitmap(skullX, skullY, alienSkull36x28, 36, 28, SSD1306_WHITE);
  display.setTextSize(1);
  display.setCursor((SCREEN_HEIGHT - 60) / 2, skullY + 30);
  display.print("SCORE: ");
  display.println(score);
  display.setCursor((SCREEN_HEIGHT - 60) / 2, skullY + 40);
  display.println("GAME OVER");
  display.setCursor((SCREEN_HEIGHT - 60) / 2, skullY + 50);
  display.println("FIRE-RETRY");
  display.display();

  while (true) {
    if (!digitalRead(BTN_FIRE)) {
      delay(300);
      inGame = false;
      break;
    }
  }
}

void resetGameObjects() {
  for (int i = 0; i < maxBullets; i++) bullets[i].active = false;
  for (int i = 0; i < maxEnemies; i++) enemies[i].active = false;
}

void loop() {
  if (!inGame) showMenu();
  if (gameOver) showGameOverScreen();

  handleInput();
  updateBullets();
  spawnEnemy();
  updateEnemies();
  checkCollisions();

  display.clearDisplay();
  drawPlayer();
  drawBullets();
  drawEnemies();
  drawHUD();
  drawDamageEffect();
  display.display();

  delay(20);
}
