#include <SPI.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <EEPROM.h>

#define SCREEN_ADDRESS 0x3C
#define BUTTON_FLAP 8
#define BUTTON_EASY 12
#define BUTTON_MEDIUM 11
#define BUTTON_HARD 10

Adafruit_SSD1306 display(128, 64, &Wire, -1);

// ==== Gracz ====
const int birdX = 20;
int birdY = 32;
int birdVelocity = 0;
const int gravity = 1;
const int flapStrength = -4;

// Animacja ptaka
const byte birdWingStates[3][5] = {
  {0,1,1,1,0},  // Skrzydła w górze
  {1,0,1,0,1},  // Skrzydła w środku
  {1,1,0,1,1}   // Skrzydła w dole
};
int currentWingState = 1;
unsigned long lastWingFlap = 0;
const int wingFlapInterval = 150;

// ==== Rury ====
const int pipeWidth = 15;
int gapHeight = 22;
const int minGapHeight = 10;

float pipeSpeed = 2.0;
float basePipeSpeed = 2.0; // Bazowa prędkość
unsigned long pipeInterval = 2000;
unsigned long lastPipeTime = 0;

struct Pipe {
  int x;
  int gapY;
  bool active;
};

const int maxPipes = 4;
Pipe pipes[maxPipes];

// ==== Monety ====
struct Coin {
  int x;
  int y;
  bool active;
  int type; // 0: srebrna (1pkt), 1: złota (2pkt)
};

const int maxCoins = 2;
Coin coins[maxCoins];
unsigned long lastCoinTime = 0;
unsigned long coinInterval = 3000;
unsigned long coinEffectEndTime = 0;
const unsigned long coinEffectDuration = 2000; // 2 sekundy spowolnienia
float coinSlowFactor = 0.7;

// ==== Power-upy ====
enum PowerUpType { NONE, SHIELD, DOUBLE_POINTS };
struct PowerUp {
  int x;
  int y;
  PowerUpType type;
  bool active;
};

const int maxPowerUps = 1;
PowerUp powerUps[maxPowerUps];
unsigned long powerUpEffectEndTime = 0;
const unsigned long powerUpDuration = 5000;
bool hasShield = false;
bool doublePoints = false;

// ==== Gra ====
bool gameOver = false;
bool gameStarted = false;
int score = 0;
int highScore = 0;
unsigned long lastFlapTime = 0;
const unsigned long flapCooldown = 150;

// ==== Poziomy trudności ====
enum Difficulty { EASY, MEDIUM, HARD };
Difficulty currentDifficulty = MEDIUM;

void setup() {
  pinMode(BUTTON_FLAP, INPUT_PULLUP);
  pinMode(BUTTON_EASY, INPUT_PULLUP);
  pinMode(BUTTON_MEDIUM, INPUT_PULLUP);
  pinMode(BUTTON_HARD, INPUT_PULLUP);
  
  Serial.begin(9600);

 if (!display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) {
  Serial.println(F("SSD1306 allocation failed"));
  for (;;);
}

display.setRotation(2); //  Odwróć ekran o 180 stopni

  // Wczytaj rekord z EEPROM
  EEPROM.get(0, highScore);
  if (highScore < 0 || highScore > 1000) highScore = 0; // Sprawdź poprawność

  randomSeed(analogRead(0));
  showMenu();
}

void loop() {
  if (!gameStarted) {
    handleMenu();
    return;
  }

  if (!gameOver) {
    unsigned long now = millis();

    // Oblicz aktualną prędkość (uwzględniając efekty spowolnienia)
    float currentSpeed = basePipeSpeed;
    if (now < coinEffectEndTime) {
      currentSpeed *= coinSlowFactor;
    }

    // Skok
    if (digitalRead(BUTTON_FLAP) == LOW && now - lastFlapTime > flapCooldown) {
      birdVelocity = flapStrength;
      lastFlapTime = now;
      currentWingState = 0; // Skrzydła w górze przy skoku
      lastWingFlap = now;
    }

    // Animacja skrzydeł
    if (now - lastWingFlap > wingFlapInterval) {
      currentWingState = (currentWingState + 1) % 3;
      lastWingFlap = now;
    }

    // Fizyka ptaka
    birdVelocity += gravity;
    birdY += birdVelocity;

    // Spawn rur
    if (now - lastPipeTime > pipeInterval) {
      spawnPipe();
      lastPipeTime = now;
      
      // 15% szans na power-up przy nowej rurze
      if (random(100) < 15 && !powerUps[0].active) {
        spawnPowerUp();
      }
    }

    // Spawn monet
    if (now - lastCoinTime > coinInterval && random(100) < 40) {
      spawnCoin();
      lastCoinTime = now;
    }

    display.clearDisplay();

    // Rysuj ptaka z animacją
    drawBird(birdX, birdY);

    // Obsługa rur
    for (int i = 0; i < maxPipes; i++) {
      if (pipes[i].active) {
        pipes[i].x -= currentSpeed;

        // Rysuj rurę
        display.fillRect(pipes[i].x, 0, pipeWidth, pipes[i].gapY, SSD1306_WHITE);
        display.fillRect(pipes[i].x, pipes[i].gapY + gapHeight, pipeWidth, 
                        64 - (pipes[i].gapY + gapHeight), SSD1306_WHITE);

        // Kolizja (jeśli nie ma tarczy)
        if (!hasShield && checkCollision(birdX, birdY, 5, 5, 
                                       pipes[i].x, 0, pipeWidth, pipes[i].gapY) ||
            checkCollision(birdX, birdY, 5, 5, 
                          pipes[i].x, pipes[i].gapY + gapHeight, 
                          pipeWidth, 64 - (pipes[i].gapY + gapHeight))) {
          gameOver = true;
          if (score > highScore) {
            highScore = score;
            EEPROM.put(0, highScore); // Zapisz nowy rekord
          }
        }

        // W funkcji loop(), w sekcji obsługi rur:

if (pipes[i].x + pipeWidth < 0) {
  pipes[i].active = false;
  score += doublePoints ? 2 : 1;

  // Zwiększ trudność co 5 punktów - TU MOŻESZ DOSTOSOWAĆ
  if (score % 5 == 0) {
    pipeSpeed += 0.2;         // Zwiększ prędkość
    if (gapHeight > minGapHeight) gapHeight -= 1; // Zmniejsz przerwy
    pipeInterval = max(pipeInterval - 50, 800);   // Częstsze rury (min 800ms)
  }
}
      }
    }

    // Obsługa monet
    for (int i = 0; i < maxCoins; i++) {
      if (coins[i].active) {
        coins[i].x -= currentSpeed;
        
        // Rysuj monetę
        drawCoin(coins[i].x, coins[i].y, coins[i].type);
        
        // Kolizja z monetą
        if (checkCollision(birdX, birdY, 5, 5, 
                          coins[i].x-2, coins[i].y-2, 4, 4)) {
          coins[i].active = false;
          score += (coins[i].type == 0) ? 1 : 2;
          if (doublePoints) score += (coins[i].type == 0) ? 1 : 2;
          coinEffectEndTime = now + coinEffectDuration; // Aktywuj spowolnienie
        }
        
        // Moneta wyleciała poza ekran
        if (coins[i].x + 4 < 0) {
          coins[i].active = false;
        }
      }
    }

    // Obsługa power-upów
    for (int i = 0; i < maxPowerUps; i++) {
      if (powerUps[i].active) {
        powerUps[i].x -= currentSpeed;
        
        // Rysuj power-up
        drawPowerUp(powerUps[i].x, powerUps[i].y, powerUps[i].type);
        
        // Kolizja z power-upem
        if (checkCollision(birdX, birdY, 5, 5, 
                          powerUps[i].x-4, powerUps[i].y-4, 8, 8)) {
          activatePowerUp(powerUps[i].type);
          powerUps[i].active = false;
        }
        
        // Power-up wyleciał poza ekran
        if (powerUps[i].x + 8 < 0) {
          powerUps[i].active = false;
        }
      }
    }

    // Rysuj tarczę jeśli aktywna
    if (hasShield && now < powerUpEffectEndTime) {
      display.drawCircle(birdX+2, birdY+2, 7, SSD1306_WHITE);
    } else {
      hasShield = false;
    }

    // Sprawdź czy power-up wygasł
    if (now > powerUpEffectEndTime) {
      doublePoints = false;
    }

    // Efekt spowolnienia (SLOW!)
    if (now < coinEffectEndTime) {
      display.setCursor(100, 0);
      display.print("SLOW");
    }

    // Kolizja z ziemią/sufitem
    if (!hasShield && (birdY < 0 || birdY + 5 > 64)) {
      gameOver = true;
      if (score > highScore) {
        highScore = score;
        EEPROM.put(0, highScore); // Zapisz nowy rekord
      }
    }

// W głównej pętli gry (loop()) zmień sekcję wyświetlania informacji na:

// Wyświetl wynik (góra ekranu, lewa strona)
display.setCursor(0, 0);
display.setTextSize(1);
display.setTextColor(SSD1306_WHITE);
display.print("Pkt:");
display.print(score);

// Wyświetl rekord (góra ekranu, środek)
display.setCursor(50, 0);
display.print("Rek:");
display.print(highScore);

// Efekt spowolnienia (góra ekranu, prawa strona)
if (now < coinEffectEndTime) {
  display.setCursor(100, 0);
  display.print("SLOW");
}

// Power-upy (linia pod wynikiem)
if (now < powerUpEffectEndTime) {
  display.setCursor(0, 10); // Druga linia
  if (hasShield) display.print("[SHIELD]");
  else if (doublePoints) display.print("[2xPTS]");
}

    display.display();
    delay(30);
  } else {
    // Ekran końca gry
    display.clearDisplay();
    display.setCursor(30, 10);
    display.setTextSize(1);
    display.setTextColor(SSD1306_WHITE);
    display.print("Koniec gry!");
    display.setCursor(20, 25);
    display.print("Wynik: ");
    display.print(score);
    display.setCursor(20, 35);
    display.print("Rekord: ");
    display.print(highScore);
    
    if (score == highScore && score > 0) {
      display.setCursor(15, 45);
      display.print("NOWY REKORD!");
    }
    
    display.setCursor(5, 55);
    display.print("Nacisnij FLAP");
    display.display();

    if (digitalRead(BUTTON_FLAP) == LOW) {
      delay(300);
      resetGame();
      showMenu();
      gameStarted = false;
    }
  }
}
void drawBird(int x, int y) {
  // Ciało ptaka
  display.fillRect(x, y, 5, 5, SSD1306_WHITE);
  
  // Skrzydła (animowane)
  for (int i = 0; i < 5; i++) {
    if (birdWingStates[currentWingState][i]) {
      display.drawPixel(x-1, y+i, SSD1306_WHITE);
      display.drawPixel(x+5, y+i, SSD1306_WHITE);
    }
  }
  
  // Oko
  display.drawPixel(x+3, y+1, SSD1306_BLACK);
}

void drawCoin(int x, int y, int type) {
  if (type == 0) { // Srebrna
    display.fillCircle(x, y, 2, SSD1306_WHITE);
  } else { // Złota
    display.fillCircle(x, y, 2, SSD1306_WHITE);
    display.drawPixel(x, y, SSD1306_BLACK);
  }
}

void drawPowerUp(int x, int y, PowerUpType type) {
  switch(type) {
    case SHIELD:
      display.drawTriangle(x, y+4, x+4, y, x+8, y+4, SSD1306_WHITE);
      display.drawTriangle(x, y+4, x+4, y+8, x+8, y+4, SSD1306_WHITE);
      break;
    case DOUBLE_POINTS:
      display.fillRect(x, y+2, 8, 4, SSD1306_WHITE);
      display.drawLine(x+1, y+1, x+7, y+7, SSD1306_BLACK);
      display.drawLine(x+1, y+7, x+7, y+1, SSD1306_BLACK);
      break;
    default: break;
  }
}

void spawnPipe() {
  for (int i = 0; i < maxPipes; i++) {
    if (!pipes[i].active) {
      pipes[i].x = 128;
      pipes[i].gapY = random(10, 64 - gapHeight - 10);
      pipes[i].active = true;
      break;
    }
  }
}

void spawnCoin() {
  for (int i = 0; i < maxCoins; i++) {
    if (!coins[i].active) {
      coins[i].x = 128;
      coins[i].y = random(10, 54);
      coins[i].type = random(100) < 70 ? 0 : 1; // 70% szans na srebrną, 30% na złotą
      coins[i].active = true;
      break;
    }
  }
}

void spawnPowerUp() {
  for (int i = 0; i < maxPowerUps; i++) {
    if (!powerUps[i].active) {
      powerUps[i].x = 128;
      powerUps[i].y = random(10, 54);
      powerUps[i].type = random(100) < 50 ? SHIELD : DOUBLE_POINTS;
      powerUps[i].active = true;
      break;
    }
  }
}

void activatePowerUp(PowerUpType type) {
  powerUpEffectEndTime = millis() + powerUpDuration;
  switch(type) {
    case SHIELD:
      hasShield = true;
      doublePoints = false;
      break;
    case DOUBLE_POINTS:
      doublePoints = true;
      hasShield = false;
      break;
    default: break;
  }
}

bool checkCollision(int x1, int y1, int w1, int h1, int x2, int y2, int w2, int h2) {
  return x1 < x2 + w2 && x1 + w1 > x2 && y1 < y2 + h2 && y1 + h1 > y2;
}

void resetGame() {
  birdY = 32;
  birdVelocity = 0;
  score = 0;
  gameOver = false;
  powerUpEffectEndTime = 0;
  hasShield = false;
  doublePoints = false;

  switch (currentDifficulty) {
  case EASY:
    basePipeSpeed = 1.2;
    pipeSpeed = 1.2;
    gapHeight = 26;
    pipeInterval = 2200;
    coinInterval = 3500;
    coinSlowFactor = 0.85;  // <<< Tutaj
    break;
  case MEDIUM:
    basePipeSpeed = 1.8;
    pipeSpeed = 1.8;
    gapHeight = 20;
    pipeInterval = 1600;
    coinInterval = 3000;
    coinSlowFactor = 0.7;  // <<< Tutaj
    break;
  case HARD:
    basePipeSpeed = 2.4;
    pipeSpeed = 2.4;
    gapHeight = 17;
    pipeInterval = 1000;
    coinInterval = 2500;
    coinSlowFactor = 0.7;  // <<< Tutaj
    break;
}
  

  for (int i = 0; i < maxPipes; i++) {
    pipes[i].active = false;
  }

  for (int i = 0; i < maxCoins; i++) {
    coins[i].active = false;
  }

  for (int i = 0; i < maxPowerUps; i++) {
    powerUps[i].active = false;
  }

  spawnPipe();
  lastPipeTime = millis();
  lastCoinTime = millis();
}

void showMenu() {
  display.clearDisplay();
  display.setCursor(30, 10);
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.print("FLAPPY BIRD");
  
  display.setCursor(20, 25);
  display.print("Trudnosc:");
  
  display.setCursor(30, 35);
  switch (currentDifficulty) {
    case EASY: 
      display.print("LATWY (SW1)");
      break;
    case MEDIUM: 
      display.print("SREDNI (SW2)");
      break;
    case HARD: 
      display.print("TRUDNY (SW3)");
      break;
  }
  
  display.setCursor(10, 50);
  display.print("FLAP-start");
  display.display();
}

void handleMenu() {
  static unsigned long lastButtonPress = 0;
  
  if (millis() - lastButtonPress > 200) {
    if (digitalRead(BUTTON_EASY) == LOW) {
      currentDifficulty = EASY;
      showMenu();
      lastButtonPress = millis();
    } 
    else if (digitalRead(BUTTON_MEDIUM) == LOW) {
      currentDifficulty = MEDIUM;
      showMenu();
      lastButtonPress = millis();
    } 
    else if (digitalRead(BUTTON_HARD) == LOW) {
      currentDifficulty = HARD;
      showMenu();
      lastButtonPress = millis();
    } 
    else if (digitalRead(BUTTON_FLAP) == LOW) {
      gameStarted = true;
      resetGame();
      lastButtonPress = millis();
    }
  }
}